#include "stdafx.h"
#include "cMemoryDC1.h" 
#include <math.h> // for floor function.  

//------------------The cMemoryDC Methods-------------------

cMemoryDC::cMemoryDC(CWnd *pWnd, COLORREF blankcol) //constructor
{
	CDC *pDC;
	int success;
 /* SM_C?FULLSCREEN gives the actual size of a fullscreen client window's
screen measured in pixels.  This assumes the window has a caption.  And we
do subtract off the height of the menu.  
	Note that we don't really need a :: for "global scope" in front of
GetSystemMetrics, as CDC has no such method.  But we make a habit of putting
:: in front of all global funtion calls.*/
	_cx = ::GetSystemMetrics(SM_CXFULLSCREEN); 
	_cy = ::GetSystemMetrics(SM_CYFULLSCREEN) - GetSystemMetrics(SM_CYMENU); 
	//Create the blankbrush.
	_cBrush_blank.CreateSolidBrush(blankcol);
	//Get a CDC* from the physical screen window.
	 pDC = pWnd->GetDC();
	CreateCompatibleDC(pDC); //Starts with area of one pixel!
	//You must use a physical screen CDC* as an argument to CreateCompatible Bitmap.
	success = _cBitmap.CreateCompatibleBitmap(pDC, _cx, _cy);
	//You don't need the physical CDC* anymore so release it right now.
	pWnd->ReleaseDC(pDC);
	//CreateCompatible Bitmap is really a memory allocation call as bitmap has size,
	//so it might very well have failed.
	if (!success) //If there wasn't enough memory, tell & bail.
	{
		pWnd->MessageBox(
			(LPSTR)"Failure in Bitmap Allocation!",
			(LPSTR)"Memory Problems!",
			MB_OK | MB_ICONEXCLAMATION ); //Button flags
		pWnd->SendMessage(WM_DESTROY, 0, 0L );
		return;
	}
	//Selecting _cBitmap here makes _hdc have _cx by _cy area.
	SelectObject(&_cBitmap);
	clear();
}

cMemoryDC::~cMemoryDC() //destructor
{
	/* The destructor will first destroy the member objects and then destroy
	the object itself.  When the CGdiObject destructor gets called for
	_cBitmap and for cBrush_blank, their DeleteObject functions are automatically
	called.  Now, there is a rule that you can't delete a GDI tool while it
	is selected into a valid HDC, so we need to get rid of the HDC wrapped
	inside of our caller cMemoryDC object.  We call DeleteDC to do this, and
	then leave the destruction of _cBitmap and _cBrush_blank to the 
	CGdiObject destructor, as already mentioned. */
	DeleteDC();
}

void cMemoryDC::clear()
{ //This clears the bitmap.
	CBrush *pcallerbrush;

	//Select the blankbrush.
	pcallerbrush = SelectObject(&_cBrush_blank);
	//This covers the screen with the selected brush pattern.
	PatBlt(0, 0, _cx, _cy, PATCOPY);
	//Reselect the callerbrush, in case this  matters.
	SelectObject(pcallerbrush);
}

void cMemoryDC::copyTo(CDC *pDC, CRect *pRect)
{
	CRect rect;
	if (pRect)
		rect = *pRect;
	else
		rect = CRect(0, 0, _cx, _cy);
	copyTo(pDC, rect);
}

void cMemoryDC::copyTo(CDC *pDC, const CRect &rect)
{
	pDC->BitBlt(rect.left, rect.top, //Next two arguments are width and height
		rect.right - rect.left, rect.bottom - rect.top,
		this, rect.left, rect.top, SRCCOPY);
}


void cMemoryDC::stretchTo(CDC *pDC, const CRect &rect, int nFlags)
{
	/* When the pDC is for print or print preview, it has a clip box rect of
	something like (-10, -10, 2000, 3000), and what happens if you do CopyTo
	is that you don't see anything at all.  Maybe your image gets squeezed up
	near (-10, -10), which isn't even visible in the print preview (off the
	upper left corner of the page). So we use a StretchBlt instead of the 
	BitBlt used by cMemoryDC::CopyTo.*/
	int target_cx;
	int target_cy;

	if (nFlags & ST_NON_ISO)
		pDC->StretchBlt(rect.left, rect.top,
			rect.right-rect.left, rect.bottom-rect.top,
			this, 0, 0, _cx, _cy, SRCCOPY);
	else
	{
		adjustStretch(_cx, _cy, rect.right-rect.left,
			rect.bottom-rect.top, target_cx, target_cy, nFlags);
		pDC->StretchBlt(rect.left, rect.top, target_cx, target_cy,
			this, 0, 0, _cx, _cy, SRCCOPY);
	}
}
//------------------The Helper Function------------------------
/* The ceiling of x is the smallest integer >= x.  We use this helper's helper 
function in adjustStretch.  It's defined usind the C library floor functoin,
which is the greatest integerg <= x. floor and ceilng come into play in
adjustStretch of the ST_INTEGER flag is on.*/

double ceiling(double x)
{
	if (x == (int)x)
		return x;
	return floor(x + 1.0);
}

void adjustStretch(int source_cx, int source_cy, int target_cx,
		int target_cy, int &stretch_target_cx, int &stretch_target_cy, 
		int nFlags)
{
	double ratio_x, ratio_y;

	
	if (!source_cx)
		source_cx = 1;
	if (!source_cy)
		source_cy = 1;
	ratio_x = ((double)target_cx) / source_cx; //cast to double or int division gives 0!
	ratio_y = ((double)target_cy) / source_cy;
	/* If no ST_*_ISO flags or on, or ST_NON_ISO is on, we leave
		the x and y the same.  */ 
	if ( !(nFlags & ST_NON_ISO) && (nFlags & ST_ISO_COVER_TARGET || nFlags & ST_ISO_USE_WHOLE_SOURCE) )
	{ //If both BIG and SMALL are on, choose BIG.
		if (nFlags & ST_ISO_COVER_TARGET)
			ratio_x = ratio_y = max(ratio_x, ratio_y);
		else 
			ratio_x = ratio_y = min(ratio_x, ratio_y);
	}
	if (nFlags & ST_INTEGER)
	{
		//Do for x
		if (ratio_x >= 1)
		{
			if (nFlags & ST_ISO_COVER_TARGET)
				ratio_x = ceiling(ratio_x);
			else
				ratio_x = floor(ratio_x);
		}
		else //ratio_x < 1
		{
			if (nFlags & ST_ISO_COVER_TARGET)
				ratio_x = 1.0 / (floor(1.0 / ratio_x));
			else
				ratio_x = 1.0 / (ceiling(1.0 / ratio_x));
		}
		//Repeat for y in case you are in the ST_NON_ISO case
		if (ratio_y >= 1)
		{
			if (nFlags & ST_ISO_COVER_TARGET)
				ratio_y = ceiling(ratio_y);
			else
				ratio_y = floor(ratio_y);
		}
		else //ratio_y < 1
		{
			if (nFlags & ST_ISO_COVER_TARGET)
				ratio_y = 1.0 / (floor(1.0 / ratio_y));
			else
				ratio_y = 1.0 / (ceiling(1.0 / ratio_y));
		}
	}
	stretch_target_cx = (int)(ratio_x * source_cx);
	stretch_target_cy = (int)(ratio_y * source_cy);
}

